/**
 * filter-nocond.c
 *
 * Дана програма моделює роботу багатопотокового фільтра, який отримує вихідні дані зі 
 * стандартного потоку введення, обробляє їх і виводить результат у стандартний потік виведення. 
 * В програмі створюються 3 робочих потоки: завантажник (downloader), обробник (processor)
 * і вивантажник (uploader). Завантажник завантажує дані з потоку введення у вхідний буфер.
 * Обробник обробляє дані, завантажені у вхідний буфер, і зберігає результат у вихідному буфері.
 * Вивантажник виводить дані з вихідного буфера у потік виведення. Обробка даних в даній програмі
 * зводиться до простого їх копіювання з вхідного буфера у вихідний. Синхронізація робочих потоків
 * виконується за допомогою 2 м'ютексів (один використовується для синхронізації завантажника
 * і обробника, другий - для синхронізації обробника і вивантажника). Синхронізація реалізована 
 * таким чином, щоб завантажник і вивантажник мали можливість працювати одночасно. 
 */
 
#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

enum { BUFSIZE = 1024 };

/* Стани вхідного буфера */
enum { IS_DOWNLOADED_NEXT, IS_DOWNLOADED_LAST, IS_PROCESSED };
/* Стани вихідного буфера */
enum { OS_PROCESSED_NEXT, OS_PROCESSED_LAST, OS_UPLOADED };

struct buffer {
        char ibuf[BUFSIZE];     /* Вхідний буфер */
        char obuf[BUFSIZE];     /* Вихідний буфер */
        int istate, ostate;     /* Стани буферів */
        int icount, ocount;     /* Лічильники байтів у буферах */
        pthread_mutex_t imutex, omutex;
};

struct buffer buffer;           /* Буфер */
FILE *istream, *ostream;        /* Вхідний та вихідний потоки */

void init_buffer();
void *downloader(void *), *processor(void *), *uploader(void *);


int main(int argc, char **argv)
{
        pthread_t downloader_thread;
        pthread_t processor_thread;
        pthread_t uploader_thread;
        int rval;

        istream = stdin;
        ostream = stdout;

        /* Виконує ініціалізацію буфера. */
        init_buffer();

        /* Створює 3 робочих потоки: завантажник (downloader),
           обробник (processor) і вивантажник (uploader). */
        rval = pthread_create(&downloader_thread, NULL, downloader, NULL);
        if (rval != 0) {
                fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(rval));
                exit(EXIT_FAILURE);
        }
        rval = pthread_create(&processor_thread, NULL, processor, NULL);
        if (rval != 0) {
                fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(rval));
                exit(EXIT_FAILURE);
        }
        rval = pthread_create(&uploader_thread, NULL, uploader, NULL);
        if (rval != 0) {
                fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(rval));
                exit(EXIT_FAILURE);
        }

        /* Чекає завершення робочих потоків. */
        rval = pthread_join(downloader_thread, NULL);
        assert(rval == 0);
        rval = pthread_join(processor_thread, NULL);
        assert(rval == 0);
        rval = pthread_join(uploader_thread, NULL);
        assert(rval == 0);

        exit(EXIT_SUCCESS);
}

/**
 * Виконує ініціалізацію буфера
 */ 
void init_buffer()
{
        int rval;
    
        buffer.istate = IS_PROCESSED;
        buffer.ostate = OS_UPLOADED;
        buffer.icount = 0;
        buffer.ocount = 0;
        rval = pthread_mutex_init(&buffer.imutex, NULL);
        if (rval == 0)
                rval = pthread_mutex_init(&buffer.omutex, NULL);
        if (rval != 0) {
                fprintf(stderr, "Error initializing mutex: %s\n",
                                                        strerror(rval));
                exit(EXIT_FAILURE);
        }
}

/**
 * Потокова функція завантажника
 */
void *downloader(void *arg)
{
        int rval, is_finished;

        is_finished = 0;        /* Ознака завершення роботи */
        for (;;) {
                /* Чекає, доки вхідний буфер буде оброблено. */
                while (1) {
                        rval = pthread_mutex_lock(&buffer.imutex);
                        assert(rval == 0);
                        if (buffer.istate == IS_PROCESSED)
                                break;
                        rval = pthread_mutex_unlock(&buffer.imutex);
                        assert(rval == 0);           
                }
                /* Заповнює вхідний буфер даними. */
                buffer.icount = fread(buffer.ibuf, 1, BUFSIZE, istream);
                if (ferror(istream)) {
                        fprintf(stderr, "Error reading input stream: %s\n",
                                                        strerror(errno));
                        buffer.istate = IS_DOWNLOADED_LAST;
                        is_finished = 1;
                } else if (feof(istream)) {
                        buffer.istate = IS_DOWNLOADED_LAST;
                        is_finished = 1;
                } else	
                        buffer.istate = IS_DOWNLOADED_NEXT;
                /* Дозволяє обробнику приступити до роботи. */
                rval = pthread_mutex_unlock(&buffer.imutex);
                assert(rval == 0);

                if (is_finished)
                        /* Завершує роботу. */
                        return NULL;
        }
        return NULL;
}

/**
 * Обробляє дані у вхідному буфері і зберігає рузельтат у вихідному буфері
 * (зараз - просто копіює зміст вхідного буфера у вихідний буфер).
 */
void process_buffer()
{
        memcpy(buffer.obuf, buffer.ibuf, buffer.icount);
        buffer.ocount = buffer.icount;
}

/**
 * Потокова функція обробника
 */
void *processor(void *arg)
{
        int rval, is_finished;
    
        is_finished = 0;        /* Ознака завершення роботи */
        for (;;) {
                /* Чекає, доки вхідний буфер буде заповнено. */
                while (1) {
                        rval = pthread_mutex_lock(&buffer.imutex);
                        assert(rval == 0);
                        if ((buffer.istate == IS_DOWNLOADED_NEXT) ||
                                (buffer.istate == IS_DOWNLOADED_LAST))
                                break;
                        rval = pthread_mutex_unlock(&buffer.imutex);
                        assert(rval == 0);                
                }
                /* Чекає, доки вихідний буфер буде звільнено. */
                while (1) {
                        rval = pthread_mutex_lock(&buffer.omutex);
                        assert(rval == 0);
                        if (buffer.ostate == OS_UPLOADED)
                                break;
                        rval = pthread_mutex_unlock(&buffer.omutex);
                        assert(rval == 0);
                }

                /* Виконує обробку даних у вхідному буфері і зберагіє
                   результат у вихідному буфері. */
                process_buffer();
                assert((buffer.istate == IS_DOWNLOADED_NEXT) ||
                                (buffer.istate == IS_DOWNLOADED_LAST));
                if (buffer.istate == IS_DOWNLOADED_NEXT) {
                        buffer.ostate = OS_PROCESSED_NEXT;
                } else {
                        buffer.ostate = OS_PROCESSED_LAST;
                        is_finished = 1;
                }
                buffer.istate = IS_PROCESSED;

                /* Дозволяє завантажнику приступити до завантаження нової порції даних. */
                rval = pthread_mutex_unlock(&buffer.imutex);
                assert(rval == 0);
                /* Дозволяє вивантажнику вивантажити чергову порцію даних. */
                rval = pthread_mutex_unlock(&buffer.omutex);
                assert(rval == 0);

                if (is_finished)
                        /* Завершує роботу. */
                        return NULL;
        }
        return NULL;
}

/**
 * Потокова функція вивантажника
 */
void *uploader(void *arg)
{
        int rval;

        for (;;) {
                /* Чекає, доки вихідний буфер буде оновлено. */
                while (1) {
                        rval = pthread_mutex_lock(&buffer.omutex);
                        assert(rval == 0);
                        if ((buffer.ostate == OS_PROCESSED_NEXT) ||
                                (buffer.ostate == OS_PROCESSED_LAST))
                                break;
                        rval = pthread_mutex_unlock(&buffer.omutex);
                        assert(rval == 0);
                }
                /* Вивантажує зміст вихідного буфера. */
                fwrite(buffer.obuf, 1, buffer.ocount, ostream);
                if (ferror(ostream)) {
                        fprintf(stderr, "Error writing output stream:"
                                                " %s\n", strerror(errno));
                        exit(EXIT_FAILURE);
                }
                assert((buffer.ostate == OS_PROCESSED_NEXT) ||
                                (buffer.ostate == OS_PROCESSED_LAST));
                if (buffer.ostate == OS_PROCESSED_LAST)
                        /* Завершує роботу. */
                        return NULL;
                buffer.ostate = OS_UPLOADED;
                /* Дозволяє обробнику приступити до роботи. */
                rval = pthread_mutex_unlock(&buffer.omutex);
                assert(rval == 0);
        }
        return NULL;
}
